/**
  ********************************************************************************
  *
  * @file    hal_t2n.h
  * @brief   Header file of T2n driver
  *
  * @version V1.0
  * @data    22 Aug 2018
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  ********************************************************************************
  */


#ifndef __HAL_T2N_H__
#define __HAL_T2N_H__

#include <hic.h>
#include "type.h"


/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @addtogroup T2n
  * @{
  */

/**
  * @defgroup T2n_Public_Types t2n Public Types
  * @{
  */

/**
  * @brief Type of t2n
  */
typedef enum {
	T20_TYPE = 0x0,
	T21_TYPE = 0x1,
} t2n_type_t;

/**
  * @brief Mode of t2n
  */
typedef enum {
	T2N_TIMER_MODE      = 0,
	T2N_DOUBLE_PWM_MODE = 0xC,
} t2n_mode_t;

/**
  * @brief Pwm polarity
  */
typedef enum {
	T2N_PWM_ALL_HIGH       = 0x0,
	T2N_PWM0_LOW_PWM1_HIGH = 0x1,
	T2N_PWM0_HIGH_PWM1_LOW = 0x2,
	T2N_PWM_ALL_LOW        = 0x3,
} t2n_polarity_t;

/**
  * @brief T2n clock div
  */
typedef enum {
	T2N_CLK_DIV_1  = 0,
	T2N_CLK_DIV_2  = 1,
	T2N_CLK_DIV_3  = 2,
	T2N_CLK_DIV_4  = 3,
	T2N_CLK_DIV_5  = 4,
	T2N_CLK_DIV_6  = 5,
	T2N_CLK_DIV_7  = 6,
	T2N_CLK_DIV_8  = 7,
	T2N_CLK_DIV_9  = 8,
	T2N_CLK_DIV_10 = 9,
	T2N_CLK_DIV_11 = 10,
	T2N_CLK_DIV_12 = 11,
	T2N_CLK_DIV_13 = 12,
	T2N_CLK_DIV_14 = 13,
	T2N_CLK_DIV_15 = 14,
	T2N_CLK_DIV_16 = 15,
} t2n_clk_div_t;

/**
  * @brief Pwm enable
  */
typedef enum {
	PWM0_EN    = 0x1,
	PWM1_EN    = 0x2,
	PWM_ALL_EN = 0x3,
} t2n_pwmn_t;

/**
  * @brief  T2N handle Structure definition
  */
typedef struct {
	t2n_mode_t mode;
	t2n_polarity_t polarity;
	t2n_clk_div_t clk_div;
	t2n_pwmn_t pwm_en;
	uint16_t period;
	uint16_t pwm0_prec;
	uint16_t pwm1_prec;
	uint8_t pos;
} t2n_init_t;
/**
  * @}
  */

/**
  * @defgroup T2n_Private_Micros t2n private Micros
  * @{
  */
#define IS_T2N_TYPE(x)				(((x) == T20_TYPE) || \
						 ((x) == T21_TYPE))
#define IS_T2N_MODE_TYPE(x)			(((x) == T2N_TIMER_MODE) || \
						 ((x) == T2N_DOUBLE_PWM_MODE))
#define IS_T2N_POLARITY_TYPE(x)			(((x) == T2N_PWM_ALL_HIGH)   || \
						 ((x) == T2N_PWM0_LOW_PWM1_HIGH)   || \
						 ((x) == T2N_PWM0_HIGH_PWM1_LOW)   || \
						 ((x) == T2N_PWM_ALL_LOW))
#define IS_T2N_CLOCK_DIV_TYPE(x)		(((x) == T2N_CLK_DIV_1)  || \
						 ((x) == T2N_CLK_DIV_2)  || \
						 ((x) == T2N_CLK_DIV_3)  || \
						 ((x) == T2N_CLK_DIV_4)  || \
						 ((x) == T2N_CLK_DIV_5)  || \
						 ((x) == T2N_CLK_DIV_6)  || \
						 ((x) == T2N_CLK_DIV_7)  || \
						 ((x) == T2N_CLK_DIV_8)  || \
						 ((x) == T2N_CLK_DIV_9)  || \
						 ((x) == T2N_CLK_DIV_10) || \
						 ((x) == T2N_CLK_DIV_11) || \
						 ((x) == T2N_CLK_DIV_12) || \
						 ((x) == T2N_CLK_DIV_13) || \
						 ((x) == T2N_CLK_DIV_14) || \
						 ((x) == T2N_CLK_DIV_15) || \
						 ((x) == T2N_CLK_DIV_16))
#define IS_T2N_PWMN_TYPE(x)			(((x) == PWM0_EN) || \
						 ((x) == PWM1_EN) || \
						 ((x) == PWM_ALL_EN))
#define IS_T2N_BEHIND_FREQ_TYPE(x)		((x) <= 127)

/**
  * @}
  */                                                                   

/**
  * @addtogroup T2N_Public_Functions
  * @{
  */						 
hal_status_t t2n_init(t2n_type_t type, t2n_init_t *init);
void t2n_start(t2n_type_t type);
uint16_t t2n_get_value(t2n_type_t type);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#endif /* __HAL_T2N_H */
